/**
 * @license
 * Copyright 2020 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { env, PadV2, util } from '@tensorflow/tfjs-core';
import { PadProgram } from '../pad_gpu';
import { PadPackedProgram } from '../pad_packed_gpu';
import { fill } from './Fill';
export const padV2 = (args) => {
    const { inputs, backend, attrs } = args;
    const { x } = inputs;
    const { paddings, constantValue } = attrs;
    if (util.sizeFromShape(x.shape) === 0) {
        // Short-circuit the computation, since x doesn't have value, only
        // the shape is used to compute output shape to pad.
        const outputShape = paddings.map((p, i) => p[0] /* beforePad */ + x.shape[i] + p[1] /* afterPad */);
        return fill({
            backend,
            attrs: { shape: outputShape, value: constantValue, dtype: x.dtype }
        });
    }
    const program = env().getBool('WEBGL_PACK_ARRAY_OPERATIONS') ?
        new PadPackedProgram(x.shape, paddings, constantValue) :
        new PadProgram(x.shape, paddings, constantValue);
    const customValues = [[constantValue]];
    return backend.runWebGLProgram(program, [x], x.dtype, customValues);
};
export const padV2Config = {
    kernelName: PadV2,
    backendName: 'webgl',
    kernelFunc: padV2
};
//# sourceMappingURL=data:application/json;base64,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